<?php
// projekTA/api/replace_card.php
// Endpoint untuk pergantian kartu (hilang/rusak/dll)
//
// Method: POST
// Body  : id_santri, uid_baru, (opsional) alasan, biaya, catatan
// Auth  : harus login; dianjurkan role admin
//
session_set_cookie_params(['path'=>'/','httponly'=>true,'samesite'=>'Lax']);
session_start();

header('Content-Type: application/json; charset=utf-8');

require_once __DIR__ . '/config.php';

// Minimal guard: harus login
if (empty($_SESSION['user_id'])) {
  http_response_code(401);
  echo json_encode(['status'=>'error','message'=>'Unauthorized']);
  exit;
}

// (Opsional tapi dianjurkan) cek role admin bila tersedia
if (isset($_SESSION['role']) && $_SESSION['role'] !== 'admin') {
  http_response_code(403);
  echo json_encode(['status'=>'error','message'=>'Hanya admin yang boleh mengganti kartu']);
  exit;
}

// Ambil input
$id_santri = isset($_POST['id_santri']) ? (int)$_POST['id_santri'] : 0;
$uid_baru  = isset($_POST['uid_baru'])  ? strtoupper(trim($_POST['uid_baru'])) : '';
$alasan    = isset($_POST['alasan'])    ? strtolower(trim($_POST['alasan'])) : 'hilang';
$biaya     = isset($_POST['biaya'])     ? (int)$_POST['biaya'] : 0;
$catatan   = isset($_POST['catatan'])   ? trim($_POST['catatan']) : '';

if ($id_santri <= 0 || $uid_baru === '') {
  http_response_code(400);
  echo json_encode(['status'=>'error','message'=>'Parameter tidak lengkap']);
  exit;
}

// Normalisasi alasan
$allowed_alasan = ['hilang','rusak','lainnya'];
if (!in_array($alasan, $allowed_alasan, true)) $alasan = 'lainnya';

$admin_id = isset($_SESSION['user_id']) ? (int)$_SESSION['user_id'] : null;

try {
  // Mulai transaksi
  $mysqli->begin_transaction();

  // Pastikan tabel riwayat ada (aman bila dieksekusi berulang)
  $ddl = "
    CREATE TABLE IF NOT EXISTS card_replacements (
      id           INT AUTO_INCREMENT PRIMARY KEY,
      id_santri    INT NOT NULL,
      uid_lama     VARCHAR(100),
      uid_baru     VARCHAR(100) NOT NULL,
      alasan       ENUM('hilang','rusak','lainnya') DEFAULT 'hilang',
      biaya        INT DEFAULT 0,
      catatan      VARCHAR(255),
      processed_by INT,
      processed_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
      CONSTRAINT fk_cr_santri FOREIGN KEY (id_santri) REFERENCES santri(id)
        ON DELETE CASCADE ON UPDATE CASCADE,
      CONSTRAINT fk_cr_user FOREIGN KEY (processed_by) REFERENCES users(id)
        ON DELETE SET NULL ON UPDATE CASCADE,
      INDEX (uid_baru),
      INDEX (processed_at)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
  ";
  if (!$mysqli->query($ddl)) {
    throw new Exception('Gagal memastikan tabel riwayat: '.$mysqli->error);
  }

  // Lock baris santri
  $stmt = $mysqli->prepare("SELECT id, nama, rfid_uid FROM santri WHERE id = ? FOR UPDATE");
  if (!$stmt) throw new Exception('Prepare gagal: '.$mysqli->error);
  $stmt->bind_param('i', $id_santri);
  $stmt->execute();
  $res = $stmt->get_result();
  $san = $res->fetch_assoc();
  $stmt->close();

  if (!$san) {
    throw new Exception('Santri tidak ditemukan');
  }

  $uid_lama = $san['rfid_uid'];

  if (strcasecmp((string)$uid_lama, $uid_baru) === 0) {
    throw new Exception('UID baru sama dengan UID lama');
  }

  // Cek apakah UID baru sudah dipakai santri lain
  $stmt = $mysqli->prepare("SELECT id, nama FROM santri WHERE rfid_uid = ? AND id != ? LIMIT 1");
  if (!$stmt) throw new Exception('Prepare cek UID gagal: '.$mysqli->error);
  $stmt->bind_param('si', $uid_baru, $id_santri);
  $stmt->execute();
  $cek = $stmt->get_result()->fetch_assoc();
  $stmt->close();

  if ($cek) {
    throw new Exception('UID baru sudah terdaftar atas nama: '.$cek['nama'].' (ID '.$cek['id'].')');
  }

  // Simpan riwayat
  $stmt = $mysqli->prepare("\n    INSERT INTO card_replacements (id_santri, uid_lama, uid_baru, alasan, biaya, catatan, processed_by)\n    VALUES (?, ?, ?, ?, ?, ?, ?)\n  ");
  if (!$stmt) throw new Exception('Prepare riwayat gagal: '.$mysqli->error);
  $stmt->bind_param('isssisi',
    $id_santri, $uid_lama, $uid_baru, $alasan, $biaya, $catatan, $admin_id
  );
  $stmt->execute();
  $stmt->close();

  // Update santri → set UID baru
  $stmt = $mysqli->prepare("UPDATE santri SET rfid_uid = ? WHERE id = ?");
  if (!$stmt) throw new Exception('Prepare update santri gagal: '.$mysqli->error);
  $stmt->bind_param('si', $uid_baru, $id_santri);
  $stmt->execute();
  if ($stmt->affected_rows < 1) {
    // Tidak berubah apa pun (harusnya minimal 1)
    throw new Exception('Gagal memperbarui UID santri');
  }
  $stmt->close();

  // Bersihkan jembatan UID (tabel rfid_latest, bila ada baris id=1)
  $mysqli->query("UPDATE rfid_latest SET uid = NULL, seen_at = NULL WHERE id = 1");

  // Bersihkan file latest_uid.txt (bila dipakai oleh perangkat)
  $latest_file = __DIR__ . '/latest_uid.txt';
  if (is_file($latest_file)) {
    // Tulis kosong supaya tidak auto-terbaca lagi
    @file_put_contents($latest_file, "");
  }

  // Commit
  $mysqli->commit();

  echo json_encode([
    'status'  => 'ok',
    'message' => 'Kartu berhasil diganti',
    'data'    => [
      'id_santri' => $san['id'],
      'nama'      => $san['nama'],
      'uid_lama'  => $uid_lama,
      'uid_baru'  => $uid_baru,
      'alasan'    => $alasan,
      'biaya'     => $biaya
    ]
  ]);

} catch (Throwable $e) {
  // Rollback bila perlu
  if ($mysqli->errno === 0) {
    // errno==0 bukan indikator aman, tapi kita coba rollback selalu
    $mysqli->rollback();
  } else {
    $mysqli->rollback();
  }
  http_response_code(400);
  echo json_encode(['status'=>'error','message'=>$e->getMessage()]);
}

